#!/usr/bin/env python
# -*-coding:utf-8 -*-
import win32print
import win32api
import os
import sys
import json

# 结束日志标识
LOG_END_MARK = "$JM$END$"


class WinPrinter:
    """
    windows连接打印机打印pdf文件
    """

    # 打印属性
    printer_attribs = None
    # 当前选中的打印机
    current_printer = None
    # 待打印目录
    print_dir = None

    def __init__(self, options=None):
        """
        初始化
        """
        if options is None or 'print_dir' not in options:
            raise Exception('请指定待打印的目录')
        # 打印目录
        self.print_dir = options["print_dir"]

        # 使用默认打印机
        self.current_printer = win32print.GetDefaultPrinter()
        # 打印属性，此处为默认设置
        self.printer_attribs = {"DesiredAccess": win32print.PRINTER_ALL_ACCESS}

    def print(self):
        """
        打印目录下所有的pdf文件
        :return:
        """
        if self.print_dir is None:
            raise Exception('请指定待打印的目录')
        if not os.path.exists(self.print_dir) or not os.path.isdir(self.print_dir):
            raise Exception('待打印的目录不存在或不是一个目录')
        # 获取所有的pdf文件
        pdf_files = []
        for files in os.walk(self.print_dir):
            for file in files[2]:
                if (os.path.splitext(file)[1]).lower() == '.pdf':
                    pdf_files.append(files[0] + file)
        # 打印文件列表
        for pdf_file in pdf_files:
            self.__print_files(pdf_files)

    def __print_files(self, file_path):
        """
        打印文件
        :param file_path:
        :return:
        """
        if self.current_printer is None:
            raise Exception('未找到打印机，请先设置打印机。')
        # 打开打印机
        h_printer = win32print.OpenPrinter(self.current_printer, self.printer_attribs)
        # 获取文档属性
        doc_info = (os.path.basename(file_path), None, None)
        # 开始打印任务
        job_id = win32print.StartDocPrinter(h_printer, 1, doc_info)
        # 开始打印任务的一个页面
        win32api.ShellExecute(0, "print", file_path, None, ".", 0)
        # 结束打印任务
        win32print.EndDocPrinter(h_printer)


def init_args():
    """
    初始化参数
    """

    argv = sys.argv[1:]
    print('共有:', len(argv), '个参数。')
    # 批次号
    if len(argv) <= 0:
        raise Exception('请指定待打印的目录')
    options = {
        'print_dir': argv[0]
    }
    return options


if __name__ == '__main__':
    """
        入口
        """
    result = {}
    try:
        printer = WinPrinter(init_args())
        printer.print()
    except Exception as e:
        result = {
            "success": False,
            "message": str(e)
        }
    print(LOG_END_MARK + json.dumps(result) + LOG_END_MARK)
