#!/usr/bin/env python
# -*-coding:utf-8 -*-
import os
import sys
import cups
import json

# 结束日志标识
LOG_END_MARK = "$JM$END$"


class LinuxPrinter:
    """
    linux连接打印机打印pdf文件
    """

    # 打印机的key
    __PRINTER_NAME_KEY = "printer-info"
    # 打印机连接
    conn = None
    # 所有打印机
    __printers = []
    # 当前选中的打印机
    current_printer = None
    # 待打印目录
    print_dir = None

    def __init__(self, options=None):
        """
        初始化
        """
        if options is None or 'print_dir' not in options:
            raise Exception('请指定待打印的目录')
        # 打印目录
        self.print_dir = options["print_dir"]
        # 打印机连接
        self.conn = cups.Connection()
        # 所有打印机
        self.__printers = self.conn.getPrinters()
        if self.__printers is None or len(self.__printers) <= 0:
            raise Exception('未找到打印机，请先设置打印机。')
        for p in self.__printers:
            if self.__printers[p] is not None:
                self.current_printer = (p, self.__printers[p])
                break

    def get_printer_list(self):
        """
        获取打印机列表
        :return:
        """
        return self.__printers

    def print(self):
        """
        打印目录下所有的pdf文件
        :return:
        """
        if self.print_dir is None:
            raise Exception('请指定待打印的目录')
        if not os.path.exists(self.print_dir) or not os.path.isdir(self.print_dir):
            raise Exception('待打印的目录不存在或不是一个目录')
        # 获取所有的pdf文件
        pdf_files = []
        for files in os.walk(self.print_dir):
            for file in files[2]:
                if (os.path.splitext(file)[1]).lower() == '.pdf':
                    pdf_files.append(str(files[0] + "/" + file))
        # 打印文件列表
        self.__print_files(pdf_files)
        return {
            "success": True,
            "message": "打印成功"
        }

    def __print_files(self, files_path):
        """
        打印文件
        :param file_path:
        :return:
        """
        if self.current_printer is None:
            raise Exception('未找到打印机，请先设置打印机。')
        printer_name = job_title = self.current_printer[0]
        self.conn.printFiles(printer_name, files_path, job_title, {})
        self.conn.enablePrinter(printer_name)
        self.conn.acceptJobs(printer_name)


def init_args():
    """
    初始化参数
    """

    argv = sys.argv[1:]
    print('共有:', len(argv), '个参数。')
    # 批次号
    if len(argv) <= 0:
        raise Exception('请指定待打印的目录')
    options = {
        'print_dir': argv[0]
    }
    return options


if __name__ == '__main__':
    """
    入口
    """
    result = {}
    try:
        printer = LinuxPrinter(init_args())
        result = printer.print()
    except Exception as e:
        result = {
            "success": False,
            "message": str(e)
        }
    print(LOG_END_MARK + json.dumps(result) + LOG_END_MARK)
